/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode.fsdataset.impl;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.File;
import java.io.IOException;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.DF;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.hdfs.StorageType;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.server.datanode.DatanodeUtil;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsVolumeSpi;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.BlockPoolSlice;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.FsDatasetImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.RamDiskReplicaTracker;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.ReplicaMap;
import org.apache.hadoop.hdfs.server.protocol.DatanodeStorage;
import org.apache.hadoop.util.DiskChecker;

@InterfaceAudience.Private
@VisibleForTesting
public class FsVolumeImpl
implements FsVolumeSpi {
    private final FsDatasetImpl dataset;
    private final String storageID;
    private final StorageType storageType;
    private final Map<String, BlockPoolSlice> bpSlices = new ConcurrentHashMap<String, BlockPoolSlice>();
    private final File currentDir;
    private final DF usage;
    private final long reserved;
    private AtomicLong reservedForRbw;
    protected volatile long configuredCapacity;
    protected ThreadPoolExecutor cacheExecutor;

    FsVolumeImpl(FsDatasetImpl dataset, String storageID, File currentDir, Configuration conf, StorageType storageType) throws IOException {
        this.dataset = dataset;
        this.storageID = storageID;
        this.reserved = conf.getLong("dfs.datanode.du.reserved", 0L);
        this.reservedForRbw = new AtomicLong(0L);
        this.currentDir = currentDir;
        File parent = currentDir.getParentFile();
        this.usage = new DF(parent, conf);
        this.storageType = storageType;
        this.configuredCapacity = -1L;
        this.cacheExecutor = this.initializeCacheExecutor(parent);
    }

    protected ThreadPoolExecutor initializeCacheExecutor(File parent) {
        if (this.storageType.isTransient()) {
            return null;
        }
        int maxNumThreads = this.dataset.datanode.getConf().getInt("dfs.datanode.fsdatasetcache.max.threads.per.volume", 4);
        ThreadFactory workerFactory = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("FsVolumeImplWorker-" + parent.toString() + "-%d").build();
        ThreadPoolExecutor executor = new ThreadPoolExecutor(1, maxNumThreads, 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(), workerFactory);
        executor.allowCoreThreadTimeOut(true);
        return executor;
    }

    File getCurrentDir() {
        return this.currentDir;
    }

    File getRbwDir(String bpid) throws IOException {
        return this.getBlockPoolSlice(bpid).getRbwDir();
    }

    File getLazyPersistDir(String bpid) throws IOException {
        return this.getBlockPoolSlice(bpid).getLazypersistDir();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void decDfsUsed(String bpid, long value) {
        FsDatasetImpl fsDatasetImpl = this.dataset;
        synchronized (fsDatasetImpl) {
            BlockPoolSlice bp = this.bpSlices.get(bpid);
            if (bp != null) {
                bp.decDfsUsed(value);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void incDfsUsed(String bpid, long value) {
        FsDatasetImpl fsDatasetImpl = this.dataset;
        synchronized (fsDatasetImpl) {
            BlockPoolSlice bp = this.bpSlices.get(bpid);
            if (bp != null) {
                bp.incDfsUsed(value);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    long getDfsUsed() throws IOException {
        long dfsUsed = 0L;
        FsDatasetImpl fsDatasetImpl = this.dataset;
        synchronized (fsDatasetImpl) {
            for (BlockPoolSlice s : this.bpSlices.values()) {
                dfsUsed += s.getDfsUsed();
            }
        }
        return dfsUsed;
    }

    long getBlockPoolUsed(String bpid) throws IOException {
        return this.getBlockPoolSlice(bpid).getDfsUsed();
    }

    @VisibleForTesting
    public long getCapacity() {
        if (this.configuredCapacity < 0L) {
            long remaining = this.usage.getCapacity() - this.reserved;
            return remaining > 0L ? remaining : 0L;
        }
        return this.configuredCapacity;
    }

    @VisibleForTesting
    public void setCapacityForTesting(long capacity) {
        this.configuredCapacity = capacity;
    }

    @Override
    public long getAvailable() throws IOException {
        long available;
        long remaining = this.getCapacity() - this.getDfsUsed() - this.reservedForRbw.get();
        if (remaining > (available = this.usage.getAvailable())) {
            remaining = available;
        }
        return remaining > 0L ? remaining : 0L;
    }

    @VisibleForTesting
    public long getReservedForRbw() {
        return this.reservedForRbw.get();
    }

    long getReserved() {
        return this.reserved;
    }

    BlockPoolSlice getBlockPoolSlice(String bpid) throws IOException {
        BlockPoolSlice bp = this.bpSlices.get(bpid);
        if (bp == null) {
            throw new IOException("block pool " + bpid + " is not found");
        }
        return bp;
    }

    @Override
    public String getBasePath() {
        return this.currentDir.getParent();
    }

    @Override
    public boolean isTransientStorage() {
        return this.storageType.isTransient();
    }

    @Override
    public String getPath(String bpid) throws IOException {
        return this.getBlockPoolSlice(bpid).getDirectory().getAbsolutePath();
    }

    @Override
    public File getFinalizedDir(String bpid) throws IOException {
        return this.getBlockPoolSlice(bpid).getFinalizedDir();
    }

    @Override
    public String[] getBlockPoolList() {
        return this.bpSlices.keySet().toArray(new String[this.bpSlices.keySet().size()]);
    }

    File createTmpFile(String bpid, Block b) throws IOException {
        return this.getBlockPoolSlice(bpid).createTmpFile(b);
    }

    @Override
    public void reserveSpaceForRbw(long bytesToReserve) {
        if (bytesToReserve != 0L) {
            this.reservedForRbw.addAndGet(bytesToReserve);
        }
    }

    @Override
    public void releaseReservedSpace(long bytesToRelease) {
        if (bytesToRelease != 0L) {
            long newReservation;
            long oldReservation;
            do {
                if ((newReservation = (oldReservation = this.reservedForRbw.get()) - bytesToRelease) >= 0L) continue;
                newReservation = 0L;
            } while (!this.reservedForRbw.compareAndSet(oldReservation, newReservation));
        }
    }

    File createRbwFile(String bpid, Block b) throws IOException {
        this.reserveSpaceForRbw(b.getNumBytes());
        return this.getBlockPoolSlice(bpid).createRbwFile(b);
    }

    File addFinalizedBlock(String bpid, Block b, File f, long bytesReservedForRbw) throws IOException {
        this.releaseReservedSpace(bytesReservedForRbw);
        return this.getBlockPoolSlice(bpid).addBlock(b, f);
    }

    Executor getCacheExecutor() {
        return this.cacheExecutor;
    }

    void checkDirs() throws DiskChecker.DiskErrorException {
        for (BlockPoolSlice s : this.bpSlices.values()) {
            s.checkDirs();
        }
    }

    void getVolumeMap(ReplicaMap volumeMap, RamDiskReplicaTracker ramDiskReplicaMap) throws IOException {
        for (BlockPoolSlice s : this.bpSlices.values()) {
            s.getVolumeMap(volumeMap, ramDiskReplicaMap);
        }
    }

    void getVolumeMap(String bpid, ReplicaMap volumeMap, RamDiskReplicaTracker ramDiskReplicaMap) throws IOException {
        this.getBlockPoolSlice(bpid).getVolumeMap(volumeMap, ramDiskReplicaMap);
    }

    public String toString() {
        return this.currentDir.getAbsolutePath();
    }

    void shutdown() {
        if (this.cacheExecutor != null) {
            this.cacheExecutor.shutdown();
        }
        Set<Map.Entry<String, BlockPoolSlice>> set = this.bpSlices.entrySet();
        for (Map.Entry<String, BlockPoolSlice> entry : set) {
            entry.getValue().shutdown();
        }
    }

    void addBlockPool(String bpid, Configuration conf) throws IOException {
        File bpdir = new File(this.currentDir, bpid);
        BlockPoolSlice bp = new BlockPoolSlice(bpid, this, bpdir, conf);
        this.bpSlices.put(bpid, bp);
    }

    void shutdownBlockPool(String bpid) {
        BlockPoolSlice bp = this.bpSlices.get(bpid);
        if (bp != null) {
            bp.shutdown();
        }
        this.bpSlices.remove(bpid);
    }

    boolean isBPDirEmpty(String bpid) throws IOException {
        File volumeCurrentDir = this.getCurrentDir();
        File bpDir = new File(volumeCurrentDir, bpid);
        File bpCurrentDir = new File(bpDir, "current");
        File finalizedDir = new File(bpCurrentDir, "finalized");
        File rbwDir = new File(bpCurrentDir, "rbw");
        if (finalizedDir.exists() && !DatanodeUtil.dirNoFilesRecursive(finalizedDir)) {
            return false;
        }
        return !rbwDir.exists() || FileUtil.list((File)rbwDir).length == 0;
    }

    void deleteBPDirectories(String bpid, boolean force) throws IOException {
        File volumeCurrentDir = this.getCurrentDir();
        File bpDir = new File(volumeCurrentDir, bpid);
        if (!bpDir.isDirectory()) {
            return;
        }
        File tmpDir = new File(bpDir, "tmp");
        File bpCurrentDir = new File(bpDir, "current");
        File finalizedDir = new File(bpCurrentDir, "finalized");
        File lazypersistDir = new File(bpCurrentDir, "lazypersist");
        File rbwDir = new File(bpCurrentDir, "rbw");
        if (force) {
            FileUtil.fullyDelete((File)bpDir);
        } else {
            if (!rbwDir.delete()) {
                throw new IOException("Failed to delete " + rbwDir);
            }
            if (!DatanodeUtil.dirNoFilesRecursive(finalizedDir) || !FileUtil.fullyDelete((File)finalizedDir)) {
                throw new IOException("Failed to delete " + finalizedDir);
            }
            if (!(!lazypersistDir.exists() || DatanodeUtil.dirNoFilesRecursive(lazypersistDir) && FileUtil.fullyDelete((File)lazypersistDir))) {
                throw new IOException("Failed to delete " + lazypersistDir);
            }
            FileUtil.fullyDelete((File)tmpDir);
            for (File f : FileUtil.listFiles((File)bpCurrentDir)) {
                if (f.delete()) continue;
                throw new IOException("Failed to delete " + f);
            }
            if (!bpCurrentDir.delete()) {
                throw new IOException("Failed to delete " + bpCurrentDir);
            }
            for (File f : FileUtil.listFiles((File)bpDir)) {
                if (f.delete()) continue;
                throw new IOException("Failed to delete " + f);
            }
            if (!bpDir.delete()) {
                throw new IOException("Failed to delete " + bpDir);
            }
        }
    }

    @Override
    public String getStorageID() {
        return this.storageID;
    }

    @Override
    public StorageType getStorageType() {
        return this.storageType;
    }

    DatanodeStorage toDatanodeStorage() {
        return new DatanodeStorage(this.storageID, DatanodeStorage.State.NORMAL, this.storageType);
    }
}

